#region References

using System;
using System.Collections;
using System.Data;
using System.Data.SqlClient;
using System.Text;
using SPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using REPORT_TABLES = gov.va.med.vbecs.Common.DatabaseConstants.BloodUnitReportTables;
using SUMMARY_COLUMNS = gov.va.med.vbecs.Common.DatabaseConstants.BloodUnitReportSummary;
using OTHER_COLUMNS = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using gov.va.med.vbecs.Common;

#endregion

namespace gov.va.med.vbecs.DAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>David Askew</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>10/07/2004</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///	Data access component providing database read functionality for
	///	blood unit history report
	///</summary>

	#endregion

	public class BloodUnitReport
	{
		#region Methods

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="5761"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5762"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetBloodUnitsForHistoryReport
		/// </summary>
		public static DataTable GetBloodUnitsForHistoryReport()
		{
			SqlParameter[] prms = 
			{
				new SqlParameter(SPROC.GetBloodUnitsForHistoryReport.divisioncode, SqlDbType.Char)
			};

			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			return StoredProcedure.GetData(SPROC.GetBloodUnitsForHistoryReport.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="5763"> 
		///		<ExpectedInput>BloodUnit Guid, input bool parameters</ExpectedInput>
		///		<ExpectedOutput>DataSet</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5764"> 
		///		<ExpectedInput>Invalid BloodUnit Guid, input parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataSet</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetSummaryData
		/// </summary>
		public static DataSet GetHistoricalBloodUnitDataTables
		(
			Guid bloodUnitGuid,
			bool includeIncomingShipmentReport,
			bool includeOutgoingShipmentReport,
			bool includeAntigenTypingReport,
			bool includeDirectedPatientReport,
			bool includeStatusReport,
			bool includeFinanceReport,
			bool includePatientAssociationReport,
			bool includeIssueReport,
			bool includeBloodUnitTestingReport,
			bool includeCrossmatchTestingReport,
			bool includeModificationReport,
			bool includeTransfusionReport
		)
		{
			DataSet reportData = new DataSet("BloodUnitReportData");
			//
			#region Summary 

			// Always add summary data
			DataTable summary;
			DataSet temp = GetReportDataSet(bloodUnitGuid,  SPROC.GetReportBloodUnitHistorySummary.StoredProcName);
			if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
			{
				summary = temp.Tables[0];  
				//
				string status = summary.Rows[0][Common.VbecsTables.UnitStatus.UnitStatusText].ToString();
				//
				if ((bool)summary.Rows[0][Common.VbecsTables.BloodUnitStatus.QuarantineIndicator])
				{
					status = string.Concat(status, " (Quarantined)");
				}
				if ((bool)summary.Rows[0][Common.VbecsTables.BloodUnitStatus.BiohazardousWarningIndicator])
				{
					status = string.Concat(status, " (Biohazardous)");
				}
				if (!summary.Rows[0].IsNull(Common.VbecsTables.BloodUnitStatus.PresumedTransfusedIndicator))
				{
					if ((bool)summary.Rows[0][Common.VbecsTables.BloodUnitStatus.PresumedTransfusedIndicator])
					{
						status = string.Concat("Presumed ", status);
					}
				}
				// show discard status and date
				if (!summary.Rows[0].IsNull("DiscardDate"))
				{
					string dateDiscarded = Common.VBECSDateTime.FormatDateTimeString(summary.Rows[0]["DiscardDate"]);
					status = string.Concat(status, ", Date: ", dateDiscarded);
				}
				//
				summary.Columns.Remove( Common.VbecsTables.UnitStatus.UnitStatusText );
				summary.Columns.Add( Common.VbecsTables.UnitStatus.UnitStatusText, typeof(string) );
				summary.Rows[0][Common.VbecsTables.UnitStatus.UnitStatusText] = status;
				//
				// show inactivated status and date
				summary.Columns.Add( "Warning", typeof(string) );
				//
				Common.RecordStatusCode recordStatusCode = Common.Utility.GetRecordStatusCodeFromString
				(
					summary.Rows[0][Common.VbecsTables.BloodUnit.RecordStatusCode].ToString()
				);
				//
				if (recordStatusCode.Equals(Common.RecordStatusCode.Inactive))
				{
					string dateInvalidated = Common.VBECSDateTime.FormatDateTimeString(summary.Rows[0][Common.VbecsTables.BloodUnit.LastUpdateDate]);
					string warning = string.Concat("**** Blood Unit Inactivated, Date: ", dateInvalidated, " ****");
					summary.Rows[0]["Warning"] = warning;
				}
				else
				{
					summary.Rows[0]["Warning"] = string.Empty;
				}
			}
			else
			{
				summary = new System.Data.DataTable();
			}
			summary.TableName = REPORT_TABLES.BloodUnitHistorySummary;
			//
			if(temp.Tables.Count == 0 || temp.Tables[0].Rows.Count == 0)
			{
				// Need a blank row to add values
				summary.Rows.Add(summary.NewRow());			
			}
			//
			reportData.Tables.Add(summary.Copy());

			#endregion
			//
			#region Antigens

			reportData.Tables[REPORT_TABLES.BloodUnitHistorySummary].Columns.Add(OTHER_COLUMNS.BloodUnitAntigens, typeof(string));
			reportData.Tables[REPORT_TABLES.BloodUnitHistorySummary].Rows[0][OTHER_COLUMNS.BloodUnitAntigens] = string.Empty;
			//
			if(temp.Tables.Count > 1 && temp.Tables[1].Rows.Count > 0)
			{
				int antigenRowCount = temp.Tables[1].Rows.Count;
				StringBuilder antigens = new StringBuilder();
				//
				for ( int idx = 0; idx < antigenRowCount; idx++ )
				{
					//CR 2853 changing the name of antigen D to wD
					string antigenName = "";
					if (!temp.Tables[1].Rows[idx].IsNull("Antigen"))
					{
						if (temp.Tables[1].Rows[idx]["Antigen"].ToString().Trim() == "D")
							antigenName = "wD";
						else
							antigenName = temp.Tables[1].Rows[idx]["Antigen"].ToString();
					}

					antigens.Append( antigenName );
					antigens.Append( " " );
					antigens.Append( temp.Tables[1].Rows[idx]["Positive"] );
					if ( idx + 1 < antigenRowCount )
					{
						antigens.Append( ", " );
					}
				}
				//
				reportData.Tables[REPORT_TABLES.BloodUnitHistorySummary].Rows[0][OTHER_COLUMNS.BloodUnitAntigens] = antigens.ToString();
			}
#if DEBUG
            //reportData.Tables[REPORT_TABLES.BloodUnitHistorySummary].WriteXml("e:\\temp\\BloodUnitHistoryReportDataSet.xml", XmlWriteMode.WriteSchema);
#endif
			#endregion
			//
			#region Subreports

			// IncomingShipmentReport
			if(includeIncomingShipmentReport)
			{
				temp = GetReportDataSet(bloodUnitGuid, SPROC.GetReportBloodUnitIncomingShipmentHistory.StoredProcName);
				if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					// This one has an extra step because we are combining data from multiple sources
					DataTable incomingShipmentTable = GetIncomingShipmentTable(temp);
					incomingShipmentTable.TableName = REPORT_TABLES.IncomingShipmentReport;
					reportData.Tables.Add(incomingShipmentTable.Copy());
				}
#if DEBUG
           // temp.WriteXml("e:\\temp\\BloodUnitIncomingShipmentHistoryDataSet.xml", XmlWriteMode.WriteSchema);
#endif
            }
			// OutgoingShipmentReport
			if(includeOutgoingShipmentReport)
			{
				temp = GetReportDataSet(bloodUnitGuid, SPROC.GetReportBloodUnitOutgoingShipmentHistory.StoredProcName);
				if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					temp.Tables[0].TableName = REPORT_TABLES.OutgoingShipmentReport;
					reportData.Tables.Add(temp.Tables[0].Copy());
				}
#if DEBUG
            //temp.WriteXml("e:\\temp\\BloodUnitOutgoingShipmentHistoryDataSet.xml", XmlWriteMode.WriteSchema);
#endif
            }
			// AntigenTypingReport
			if(includeAntigenTypingReport)
			{
				temp = GetReportDataSet(bloodUnitGuid, SPROC.GetReportBloodUnitAnitgenTypingHistory.StoredProcName);
				if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					temp.Tables[0].TableName = REPORT_TABLES.AntigenTypingReport;

					//CR 2853 changing the name of antigen D to wD
					for (int i=0; i< temp.Tables[0].Rows.Count; i++)
					{
						if (!temp.Tables[0].Rows[i].IsNull("Antigen"))
						{
							if (temp.Tables[0].Rows[i]["Antigen"].ToString().Trim() == "D")
								temp.Tables[0].Rows[i]["Antigen"] = "wD";
						}
					}

					reportData.Tables.Add(temp.Tables[0].Copy());
				}
#if DEBUG
             //   temp.WriteXml("e:\\temp\\BloodUnitAntigenTypingHistoryDataSet.xml", XmlWriteMode.WriteSchema);
#endif
            }
			// RestrictedPatientReport
			if(includeDirectedPatientReport)
			{
				temp = GetReportDataSet(bloodUnitGuid, SPROC.GetReportBloodUnitDirectedPatientHistory.StoredProcName);
				//
				// It is possible that null patient data will be returned if there has been no patient association.  
				// Consequently, directed patient history is considered valid if at least 1 row is returned with 
				// non-null patient first and last name.
				bool validHistory = false;
				//
				if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					for (int idx = 0; idx < temp.Tables[0].Rows.Count; idx++)
					{
						if(!temp.Tables[0].Rows[idx].IsNull(TABLES.Patient.PatientLastName) && !temp.Tables[0].Rows[idx].IsNull(TABLES.Patient.PatientFirstName))
						{
							validHistory = true;
							break;
						}
					}
				}
				//
				if(validHistory)
				{
					temp.Tables[0].TableName = REPORT_TABLES.DirectedPatientReport;
					reportData.Tables.Add(temp.Tables[0].Copy());
				}
#if DEBUG
             //   temp.WriteXml("e:\\temp\\BloodUnitRestrictedPatientHistoryDataSet.xml", XmlWriteMode.WriteSchema);
#endif
            }
			// StatusReport
			if(includeStatusReport)
			{
				temp = GetReportDataSet(bloodUnitGuid, SPROC.GetReportBloodUnitStatusHistory.StoredProcName);
				if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					temp.Tables[0].TableName = REPORT_TABLES.StatusReport;
					reportData.Tables.Add(temp.Tables[0].Copy());
				}
#if DEBUG
              //  temp.WriteXml("e:\\temp\\BloodUnitStatusHistoryDataSet.xml", XmlWriteMode.WriteSchema);
#endif
            }
			// FinanceReport
			if(includeFinanceReport)
			{
				temp = GetReportDataSet(bloodUnitGuid, SPROC.GetReportBloodUnitFinanceHistory.StoredProcName);
				if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					temp.Tables[0].TableName = REPORT_TABLES.FinanceReport;
					reportData.Tables.Add(temp.Tables[0].Copy());
				}
#if DEBUG
               // temp.WriteXml("e:\\temp\\BloodUnitFinancialHistoryDataSet.xml", XmlWriteMode.WriteSchema);
#endif
            }
			// PatientAssociationReport
			if(includePatientAssociationReport)
			{
				temp = GetReportDataSet(bloodUnitGuid, SPROC.GetReportBloodUnitPatientAssociationHistory.StoredProcName);
				if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					temp.Tables[0].TableName = REPORT_TABLES.PatientAssociationReport;
					reportData.Tables.Add(temp.Tables[0].Copy());
				}
#if DEBUG
            //    temp.WriteXml("e:\\temp\\BloodUnitPatientAssociationHistoryDataSet.xml", XmlWriteMode.WriteSchema);
#endif
            }
			// IssueReport
			if(includeIssueReport)
			{
				temp = GetReportDataSet(bloodUnitGuid, SPROC.GetReportBloodUnitIssueHistory.StoredProcName);
				if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					temp.Tables[0].TableName = REPORT_TABLES.IssueReport;
					reportData.Tables.Add(temp.Tables[0].Copy());
				}
#if DEBUG
            //    temp.WriteXml("e:\\temp\\BloodUnitIssueHistoryDataSet.xml", XmlWriteMode.WriteSchema);
#endif
            }
			// BloodUnitTestingReport
			if(includeBloodUnitTestingReport)
			{
				temp = GetReportDataSet(bloodUnitGuid, SPROC.GetReportBloodUnitTestingHistory.StoredProcName);
				if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					temp.Tables[0].TableName = REPORT_TABLES.BloodUnitTestingReport;
					reportData.Tables.Add(temp.Tables[0].Copy());
				}
#if DEBUG
            //    temp.WriteXml("e:\\temp\\BloodUnitTestingHistoryDataSet.xml", XmlWriteMode.WriteSchema);
#endif
            }
			// CrossmatchTestingReport
			if(includeCrossmatchTestingReport)
			{
				temp = GetReportDataSet(bloodUnitGuid, SPROC.GetReportBloodUnitCrossmatchTestingHistory.StoredProcName);
				if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					temp.Tables[0].TableName = REPORT_TABLES.CrossmatchTestingReport;
					reportData.Tables.Add(temp.Tables[0].Copy());
				}
#if DEBUG
           //     temp.WriteXml("e:\\temp\\BloodUnitCrossmatchTestingHistoryDataSet.xml", XmlWriteMode.WriteSchema);
#endif
            }
			// ModificationReport
			if(includeModificationReport)
			{
				// First, determine the mod type
				SqlParameter[] prms =
				{
					new SqlParameter(SPROC.GetReportBloodUnitModificationSummary.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
					new SqlParameter(SPROC.GetReportBloodUnitModificationSummary.divisioncode, System.Data.SqlDbType.Char),
				};
			
				prms[0].Value = bloodUnitGuid;
				prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

				int modType =  new Common.StoredProcedure().GetIntReturnValue( SPROC.GetReportBloodUnitModificationSummary.StoredProcName, prms );

				#region Original and Target Units

				DataTable parentUnits;
				temp = GetReportDataSet(bloodUnitGuid, SPROC.GetReportBloodUnitSourceModificationSummary.StoredProcName);
				if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					parentUnits = temp.Tables[0];	
					parentUnits.TableName = REPORT_TABLES.OriginalUnitsFromModification;
					reportData.Tables.Add(parentUnits.Copy());
				}
#if DEBUG
           //     temp.WriteXml("e:\\temp\\BloodUnitModificationOriginalUnitsHistoryDataSet.xml", XmlWriteMode.WriteSchema);
#endif
	
				DataTable childUnits;
				temp = GetReportDataSet(bloodUnitGuid, SPROC.GetReportBloodUnitTargetModificationSummary.StoredProcName);
				if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					childUnits = temp.Tables[0];
					childUnits.TableName = REPORT_TABLES.TargetUnitsFromModification;
					reportData.Tables.Add(childUnits.Copy());
				}
#if DEBUG
          //      temp.WriteXml("e:\\temp\\BloodUnitModificationTargetUnitsHistoryDataSet.xml", XmlWriteMode.WriteSchema);
#endif

				#endregion
				//
				if ( modType == 0 )
				{
					temp = GetReportDataSet( bloodUnitGuid, SPROC.GetReportBloodUnitPoolModificationHistory.StoredProcName );
				}
				else
				{
					temp = GetReportDataSet( bloodUnitGuid, SPROC.GetReportBloodUnitModificationHistory.StoredProcName );
				}
#if DEBUG
          /*      temp.Tables[0].WriteXml("e:\\temp\\BloodUnitModificationHistoryDataSet.xml", XmlWriteMode.WriteSchema);
                if (temp.Tables.Count > 1)
                {
                    temp.Tables[1].WriteXml("e:\\temp\\BloodUnitModificationEquipmentHistoryDataSet.xml", XmlWriteMode.WriteSchema);
                }*/
#endif
                if (temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					temp.Tables[0].TableName = REPORT_TABLES.ModificationReport;
					reportData.Tables.Add(temp.Tables[0].Copy());
					if(temp.Tables.Count == 2 && temp.Tables[1].Rows.Count > 0)
					{
						temp.Tables[1].TableName = REPORT_TABLES.ModificationEquipmentAndSupplies;
						reportData.Tables.Add(temp.Tables[1].Copy());
					}
				}
			}
			// TransfusionReport
			if(includeTransfusionReport)
			{
				temp = GetReportDataSet(bloodUnitGuid, SPROC.GetReportBloodUnitTransfusionHistory.StoredProcName);
				if(temp.Tables.Count > 0 && temp.Tables[0].Rows.Count > 0)
				{
					temp.Tables[0].TableName = REPORT_TABLES.TransfusionReport;
					reportData.Tables.Add(temp.Tables[0].Copy());
				}
#if DEBUG
             //   temp.WriteXml("e:\\temp\\BloodUnitTransfusionHistoryDataSet.xml", XmlWriteMode.WriteSchema);
#endif
            }

			#endregion
			//
			return reportData;
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8369"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8370"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns estimated record count for the unit history report
		/// </summary>
		public static int GetReportBloodUnitHistoryRecordCount
		(
			Guid bloodUnitGuid,
			bool includeIncomingShipmentReport,
			bool includeOutgoingShipmentReport,
			bool includeAntigenTypingReport,
			bool includeRestrictedPatientReport,
			bool includeStatusReport,
			bool includeFinanceReport,
			bool includePatientAssociationReport,
			bool includeIssueReport,
			bool includeBloodUnitTestingReport,
			bool includeCrossmatchTestingReport,
			bool includeModificationReport,
			bool includeTransfusionReport,
			bool includeExceptionReport
		)
		{
			SqlParameter[] prms = 
			{
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.bloodunitguid, SqlDbType.UniqueIdentifier),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.divisioncode, SqlDbType.Char),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.includeincomingshipmentreport, SqlDbType.Bit),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.includeoutgoingshipmentreport, SqlDbType.Bit),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.includeantigentypingreport, SqlDbType.Bit),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.includerestrictedpatientreport, SqlDbType.Bit),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.includestatusreport, SqlDbType.Bit),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.includefinancereport, SqlDbType.Bit),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.includepatientassociationreport, SqlDbType.Bit),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.includeissuereport, SqlDbType.Bit),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.includebloodunittestingreport, SqlDbType.Bit),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.includecrossmatchtestingreport, SqlDbType.Bit),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.includemodificationreport, SqlDbType.Bit),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.includetransfusionreport, SqlDbType.Bit),
				new SqlParameter(SPROC.GetReportBloodUnitHistoryRecordCount.includeexceptionreport, SqlDbType.Bit)
			};
			//
			prms[0].Value = bloodUnitGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			prms[2].Value = includeIncomingShipmentReport;
			prms[3].Value = includeOutgoingShipmentReport;
			prms[4].Value = includeAntigenTypingReport;
			prms[5].Value = includeRestrictedPatientReport;
			prms[6].Value = includeStatusReport;
			prms[7].Value = includeFinanceReport;
			prms[8].Value = includePatientAssociationReport;
			prms[9].Value = includeIssueReport;
			prms[10].Value = includeBloodUnitTestingReport;
			prms[11].Value = includeCrossmatchTestingReport;
			prms[12].Value = includeModificationReport;
			prms[13].Value = includeTransfusionReport;
			prms[14].Value = includeExceptionReport;
			//
			return new Common.StoredProcedure().GetIntReturnValue(SPROC.GetReportBloodUnitHistoryRecordCount.StoredProcName, prms);
		}

		#region Helper Methods

		/// <summary>
		/// GetIncomingShipmentTable
		/// </summary>
		private static DataTable GetIncomingShipmentTable(DataSet incomingShipmentData)
		{
			DataTable incomingShipmentTable = GetEmptyIncomingShipmentTable( true );
			//
			for ( int tableIdx = 0; tableIdx < incomingShipmentData.Tables.Count; tableIdx++ )
			{
				for ( int rowIdx = 0; rowIdx < incomingShipmentData.Tables[tableIdx].Rows.Count; rowIdx++)
				{
					DataRow dr = incomingShipmentTable.NewRow();
					//
					object [] key = new object[] { incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.BloodUnit.LastUpdateDate] };
					if(!incomingShipmentTable.Rows.Contains( key ) )
					{
						#region New Row

						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.Shipment.ShipmentDate))
						{
							dr[TABLES.Shipment.ShipmentDate] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.Shipment.ShipmentDate];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.Shipment.InvoiceNumber))
						{
							dr[TABLES.Shipment.InvoiceNumber] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.Shipment.InvoiceNumber];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(OTHER_COLUMNS.ShipmentFacility))
						{
							dr[OTHER_COLUMNS.ShipmentFacility] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][OTHER_COLUMNS.ShipmentFacility];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(OTHER_COLUMNS.CollectionFacility))
						{
							dr[OTHER_COLUMNS.CollectionFacility] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][OTHER_COLUMNS.CollectionFacility];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.DonationType.DonationTypeText))
						{
							dr[TABLES.DonationType.DonationTypeText] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.DonationType.DonationTypeText];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.EntryMethod.EntryMethodText))
						{
							dr[TABLES.EntryMethod.EntryMethodText] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.EntryMethod.EntryMethodText];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.BloodUnit.ProcessingTechId))
						{
							dr[TABLES.BloodUnit.ProcessingTechId] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.BloodUnit.ProcessingTechId];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.BloodUnit.OriginalVolume))
						{
							dr[TABLES.BloodUnit.OriginalVolume] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.BloodUnit.OriginalVolume];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.BloodUnit.DiscardedPlasmaVolume))
						{
							dr[TABLES.BloodUnit.DiscardedPlasmaVolume] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.BloodUnit.DiscardedPlasmaVolume];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains("BloodTestName"))
						{
							if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx]["BloodTestName"].ToString().Equals("CMV"))
							{
								dr[OTHER_COLUMNS.CMVNegative] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx]["TestResultText"];
							}
							if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx]["BloodTestName"].ToString().Equals("Sickle Cell Interp"))
							{
								dr[OTHER_COLUMNS.SCNegative] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx]["TestResultText"];
							}
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.BloodUnit.LastUpdateUser))
						{
							dr[TABLES.BloodUnit.LastUpdateUser] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.BloodUnit.LastUpdateUser];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.BloodUnit.LastUpdateDate))
						{
							dr[TABLES.BloodUnit.LastUpdateDate] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.BloodUnit.LastUpdateDate];
						}
						//
						incomingShipmentTable.Rows.Add( dr );

						#endregion
					}
					else
					{
						#region Updated Row

						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.Shipment.ShipmentDate))
						{
							incomingShipmentTable.Rows.Find( key )[TABLES.Shipment.ShipmentDate] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.Shipment.ShipmentDate];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.Shipment.InvoiceNumber))
						{
							incomingShipmentTable.Rows.Find( key )[TABLES.Shipment.InvoiceNumber] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.Shipment.InvoiceNumber];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(OTHER_COLUMNS.ShipmentFacility))
						{
							incomingShipmentTable.Rows.Find( key )[OTHER_COLUMNS.ShipmentFacility] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][OTHER_COLUMNS.ShipmentFacility];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(OTHER_COLUMNS.CollectionFacility))
						{
							incomingShipmentTable.Rows.Find( key )[OTHER_COLUMNS.CollectionFacility] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][OTHER_COLUMNS.CollectionFacility];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.DonationType.DonationTypeText))
						{
							incomingShipmentTable.Rows.Find( key )[TABLES.DonationType.DonationTypeText] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.DonationType.DonationTypeText];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.EntryMethod.EntryMethodText))
						{
							incomingShipmentTable.Rows.Find( key )[TABLES.EntryMethod.EntryMethodText] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.EntryMethod.EntryMethodText];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.BloodUnit.ProcessingTechId))
						{
							incomingShipmentTable.Rows.Find( key )[TABLES.BloodUnit.ProcessingTechId] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.BloodUnit.ProcessingTechId];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.BloodUnit.OriginalVolume))
						{
							incomingShipmentTable.Rows.Find( key )[TABLES.BloodUnit.OriginalVolume] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.BloodUnit.OriginalVolume];
						}
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.BloodUnit.DiscardedPlasmaVolume))
						{
							incomingShipmentTable.Rows.Find( key )[TABLES.BloodUnit.DiscardedPlasmaVolume] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.BloodUnit.DiscardedPlasmaVolume];
						}	
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains("BloodTestName"))
						{
							if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx]["BloodTestName"].ToString().Equals("CMV"))
							{
								incomingShipmentTable.Rows.Find( key )[OTHER_COLUMNS.CMVNegative] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx]["TestResultText"];
							}
							if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx]["BloodTestName"].ToString().Equals("Sickle Cell Interp"))
							{
								incomingShipmentTable.Rows.Find( key )[OTHER_COLUMNS.SCNegative] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx]["TestResultText"];
							}
						}
						//
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.BloodUnit.LastUpdateUser))
						{
							incomingShipmentTable.Rows.Find( key )[TABLES.BloodUnit.LastUpdateUser] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.BloodUnit.LastUpdateUser];
						}
						//
						if(incomingShipmentData.Tables[tableIdx].Rows[rowIdx].Table.Columns.Contains(TABLES.BloodUnit.LastUpdateDate))
						{
							incomingShipmentTable.Rows.Find( key )[TABLES.BloodUnit.LastUpdateDate] = incomingShipmentData.Tables[tableIdx].Rows[rowIdx][TABLES.BloodUnit.LastUpdateDate];
						}

						#endregion
					}
				}
			}
			//
			#region Sort the data

			// Sort table by LastUpdateDate DESC
			DataRow [] sortedRows = incomingShipmentTable.Select("", "LastUpdateDate DESC");
			DataTable sortedTable = new DataTable();
			sortedTable = incomingShipmentTable.Clone();
			for( int idx = 0; idx < sortedRows.Length; idx++ )
			{
				sortedTable.ImportRow( sortedRows[idx] );
			}
			//
			// Remove rows that have no updated info and also only show 
			// column data that was updated
			// Note: we start at the bottom (earliest) and move forward
			// since we will always show the first entry
			//
			DataTable origTable = sortedTable.Copy();
			//
			int [] rowCheck = new int[sortedTable.Rows.Count];
			int rowCount = sortedTable.Rows.Count;
			//
			if (rowCount > 1)
			{
				for (int rowIdx = rowCount - 1; rowIdx > 0; rowIdx--)
				{
					for (int colIdx = 0; colIdx < origTable.Columns.Count; colIdx++)
					{
						// Ignore LastUpdateDate and LastUpdateUser
						if(!origTable.Columns[colIdx].ColumnName.Equals(TABLES.BloodUnit.LastUpdateDate)
							&& !origTable.Columns[colIdx].ColumnName.Equals(TABLES.BloodUnit.LastUpdateUser))
						{
							if(origTable.Rows[rowIdx - 1][origTable.Columns[colIdx]].Equals(
								origTable.Rows[rowIdx][origTable.Columns[colIdx]]))
							{
								// If there's an invoice, then this must be a new shipment record, 
								// so we show all the columns; otherwise, edits are filtered 
								// for what was changed.
								if(origTable.Rows[rowIdx - 1].IsNull(TABLES.Shipment.InvoiceNumber))
								{
									sortedTable.Rows[rowIdx - 1][sortedTable.Columns[colIdx]] = DBNull.Value;
								}
							}
						}
					}
				}
			}
			//
			// Last time through; now remove the blank rows
			rowCount = sortedTable.Rows.Count;
			for (int rowIdx = rowCount - 1; rowIdx > 0; rowIdx--)
			{
				bool nullRow = true;
				//
				for (int colIdx = 0; colIdx < sortedTable.Columns.Count; colIdx++)
				{
					// Ignore LastUpdateDate and LastUpdateUser
					if(!sortedTable.Columns[colIdx].ColumnName.Equals(TABLES.BloodUnit.LastUpdateDate)
						&& !sortedTable.Columns[colIdx].ColumnName.Equals(TABLES.BloodUnit.LastUpdateUser))
					{
						if(!sortedTable.Rows[rowIdx - 1].IsNull(sortedTable.Columns[colIdx]))
						{
							nullRow = false;
							break;
						}
					}
				}
				//
				if (nullRow)
				{
					sortedTable.Rows.Remove(sortedTable.Rows[rowIdx - 1]);
				}
			}

			#endregion
			//
			return sortedTable;
		}

		/// <summary>
		/// GetEmptyIncomingShipmentTable
		/// </summary>
		private static DataTable GetEmptyIncomingShipmentTable(bool createPrimaryKey)
		{
			DataTable incomingShipmentTable = new DataTable();
			//
			incomingShipmentTable.Columns.Add(TABLES.Shipment.ShipmentDate, typeof(DateTime));
			incomingShipmentTable.Columns.Add(TABLES.Shipment.InvoiceNumber, typeof(string));
			incomingShipmentTable.Columns.Add(OTHER_COLUMNS.ShipmentFacility, typeof(string));
			incomingShipmentTable.Columns.Add(OTHER_COLUMNS.CollectionFacility, typeof(string));
			incomingShipmentTable.Columns.Add(TABLES.DonationType.DonationTypeText, typeof(string));
			incomingShipmentTable.Columns.Add(TABLES.EntryMethod.EntryMethodText, typeof(string));
			incomingShipmentTable.Columns.Add(TABLES.BloodUnit.ProcessingTechId, typeof(string));
			incomingShipmentTable.Columns.Add(TABLES.BloodUnit.OriginalVolume, typeof(int));
			incomingShipmentTable.Columns.Add(TABLES.BloodUnit.DiscardedPlasmaVolume, typeof(int));
			incomingShipmentTable.Columns.Add(OTHER_COLUMNS.CMVNegative, typeof(string));
			incomingShipmentTable.Columns.Add(OTHER_COLUMNS.SCNegative , typeof(string));
			incomingShipmentTable.Columns.Add(TABLES.BloodUnit.LastUpdateUser, typeof(string));
			incomingShipmentTable.Columns.Add(TABLES.BloodUnit.LastUpdateDate, typeof(DateTime));
			//
			if(createPrimaryKey)
			{
				DataColumn [] keys = new DataColumn []
				{
					incomingShipmentTable.Columns[TABLES.BloodUnit.LastUpdateDate]
				};
				incomingShipmentTable.PrimaryKey = keys;
			}
			//
			return incomingShipmentTable;
		}

		/// <summary>
		/// GetReportDataSet
		/// </summary>
		private static DataSet GetReportDataSet(Guid bloodUnitGuid, string storedProcedureName)
		{
			SqlParameter[] prms = new SqlParameter[2]
				{
					new SqlParameter(SPROC.GetReportBloodUnitIncomingShipmentHistory.bloodunitguid, SqlDbType.UniqueIdentifier),
					new SqlParameter(SPROC.GetReportBloodUnitIncomingShipmentHistory.divisioncode, SqlDbType.VarChar)
				};

			prms[0].Value = bloodUnitGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return StoredProcedure.GetData(storedProcedureName, prms);
		}

		#endregion

		#endregion
	}
}